      program sacevalr 
c---------------------------------------------------------------------c 
c                                                                    c 
c     COMPUTER PROGRAMS IN SEISMOLOGY                                c 
c     VOLUME VIII                                                    c 
c                                                                    c
c     PROGRAM: SACEVALR                                              c
c                                                                    c
c     COPYRIGHT 2000                                                 c
c     R. B. Herrmann                                                 c
c     Department of Earth and Atmospheric Sciences                   c
c     Saint Louis University                                         c
c     221 North Grand Boulevard                                      c
c     St. Louis, Missouri 63103                                      c
c     U. S. A.                                                       c
c                                                                    c
c---------------------------------------------------------------------c
c-----
c      CHANGES
c      19 MAR 2001 Built in code to look for 360 degree jumps in 
c              phase in the EVALRESP PHASE file
c      21 MAY 2002 When response is removed, set the MINPER MAXPER
c              on the basis of the water level
c      01 APR 2004 The depmax, depmin and depmen of the output
c              sac file is not correctly set
c      23 MAY 2004 Added FREQLIMITS from GSAC and also permit NPOLES
c              with no poles listed in the manner of SAC
c      04 NOV 2004 correctly set permin permax in user1 and user2
c      11 NOV 2005 got rid of incorrect call to at line97
c                      if(ierr.lt.0)then
c                         call perr('Error reading the response files')
c                  endif
c              since ierr was not set anywhhere
c       23 JUL 2009 - ensured consistent call to FFT routine with
c               complex arguments. We now call zfour
c-----
c-----
c
c    sacevalr -DEMEAN -TAPER -FREQLIMITS f1 f2 f3 f4 -A -R 
c      -AMP amp_file -PHA phase_file -SACIN binary_sac_in
c      -SACOUT binary_sac_out 
c
c    This program applies or removes a complex instrument response
c      given by evalresp(IRIS). The input and output files
c      are SAC binary files.  
c-----
c      command line arguments
c-----
        logical domean, dotape, doappl
        real*4 f1, f2, f3, f4
        character famp*80, fpha*80, sacin*80, sacout*80
        logical doftaper
c-----
c      variables local to the program
c-----
        integer LER, LIN, LOT
        parameter (LER=0, LIN=5, LOT=6)
        parameter (LN=131072)
        integer*4 npts
        real*4 x(LN)
        real depmax, depmin, depmen
c-----
c      SAC header files
c-----
        common/sachdr/rhdr,ihdr,chdr
        real*4 rhdr(70)
        integer*4 ihdr(40)
        character*8 chdr(24)
c-----
c      complex response values
c-----
        parameter (NRESP=20000)
        real tamp(NRESP), tpha(NRESP)
        integer nfreq
        real tfreq(NRESP)
c-----
c      call machine dependent initialization
c-----
        call mchdep()
c-----
c      parse command line arguments
c-----
        call gcmdln(domean, dotape, doappl, f1, f2, f3, f4,
     1      sacin, sacout, famp, fpha, doftaper)
        write(LER,*)'domean :',domean
        write(LER,*)'dotape :',dotape
        write(LER,*)'doappl :',doappl
        if(doftaper)then
            write(LER,*)'f1     :',f1    
            write(LER,*)'f2     :',f2    
            write(LER,*)'f3     :',f3    
            write(LER,*)'f4     :',f4    
        endif
        write(LER,*)'sacin  :',sacin 
        write(LER,*)'sacout :',sacout
        write(LER,*)'famp   :',famp  
        write(LER,*)'fpha   :',fpha  
c-----
c      get the complex response for the instrument 
c      by reading separate amplitude(freq) and phase(freq) 
c      tables generated by evalresp(IRIS)
c-----
        call gettamp(famp,fpha,nfreq,tfreq,tamp,tpha,NRESP)
c-----
c      process
c-----
        IRU = 1
        call brsac (IRU,LN,sacin,x,nerr)
        if(nerr.ge.0)then
            call getnhv('NPTS',npts,nerr)
            if(nerr.lt.0)then
                call perr('Cannot get NPTS')
            endif
            call getfhv('DELTA',dt,nerr)
            fnyq = 0.5/dt

            if(domean)call dmean(x,npts)
            if(dotape)call dtape(x,npts,pct)
            call getfhv('USER1', opermin, ierr)
            call getfhv('USER2', opermax, ierr)
            call filt(doappl,x,npts,dt,f1,f2,f3,f4,doftaper,
     1          nfreq,tfreq,tamp,tpha,NRESP,permin,permax)
            call scmxmn(x,npts,depmax,depmin,depmen,indmax,indmin)
            call getfhv('B     ', btime , ierr)
            call setfhv('TIMMAX', btime + indmax*dt, ierr)
            call setfhv('TIMMIN', btime + indmin*dt, ierr)
          
            call setfhv('DEPMAX', depmax, ierr)
            call setfhv('DEPMIN', depmin, ierr)
            call setfhv('DEPMEN', depmen, ierr)
c-----
c          permin and permax are set only if the instrument
c          is removed
c-----
            if(opermin .ge. 0.0 .and. opermax .ge.0.0)then
                if(permin.lt.opermin)permin = opermin
                if(permax.gt.opermax)permin = opermax
            endif
            call setfhv('USER1', permin, ierr)
            call setfhv('USER2', permax, ierr)
            call setkhv('KUSER1', 'PER_MIN  ', ierr)
            call setkhv('KUSER2', 'PER_MAX  ', ierr)
c-----
c          output the filtered trace
c-----
            IWU = 2
            call bwsac(IWU,LN,sacout,x)
        else
            call perr('Error reading input file')
        endif
        end

        subroutine perr(ostr)
        integer LER, LIN, LOT
        parameter (LER=0, LIN=5, LOT=6)
        character ostr*(*)
        write(LER,*)ostr
        stop
        end

        subroutine gcmdln(domean, dotape, doappl, f1, f2, f3, f4,
     1      sacin, sacout, famp, fpha, doftaper)
c-----
c      sacevalr -DEMEAN -TAPER  -FREQLIMITS f1 f2 f3 f4 
c          -AMP amp_file -PHA phase_file -SACIN binary_sac_in
c          -SACOUT binary_sac_out 
c
c      parse command line arguments and return control
c      parameters
c
c      domean  L   - .true. remove mean from time seeries
c      dotape  L   - .true. cosine taper ends
c      doappl  L   - .true. apply filter
c                .false. remove filter
c      doftaper L  If true do frequency taper
c      f1  R*4 Apply zero phase band bass [f1,f2,f3,f4]
c      f2  R*4 where response = 1 for f2 <= f <= f3
c      f3  R*4
c      f4  R*4
c      sacin   Ch*80   - binary sac trace file for input
c      sacout  Ch*80   - binary sac trace file for out
c      famp    Ch*80   - ASCII (frequency, spectral amplitude) 
c                        from evalresp(IRIS)
c      fpha    Ch*80   - ASCII (frequency, spectral phase    ) 
c                        from evalresp(IRIS)
c
c-----
        logical domean, dotape, doappl
        character sacin*80, sacout*80, famp*80, fpha*80
        logical doftaper

        integer LER, LIN, LOT
        parameter (LER=0, LIN=5, LOT=6)
        
        character*80 name
        integer*4 nmarg
        nmarg = mnmarg()
        domean = .false.
        dotape = .false.
        doappl = .true.
        doftaper = .false.
        f1 = -10.
        f2 = -5.
        f3 = 1.0e+6
        f4 = 1.0e+7
        famp = ' '
        fpha = ' '
        sacin  = ' '
        sacout = ' '
        i = 0
   11   i = i + 1
            if(i.gt.nmarg)goto 13
            call mgtarg(i,name)
            if(name(1:3).eq.'-DE')then
                domean = .true.
            else if(name(1:3).eq.'-TA')then
                dotape = .true.
            else if(name(1:2).eq.'-A'.and.name(1:3).ne.'-AM')then
                doappl = .true.
            else if(name(1:2).eq.'-R')then
                doappl = .false.
            else if(name(1:3).eq.'-FR' .or. name(1:3).eq.'-fr')then
                i = i + 1
                call mgtarg(i,name)
                read(name,'(bn,f10.0)')f1
                i = i + 1
                call mgtarg(i,name)
                read(name,'(bn,f10.0)')f2
                i = i + 1
                call mgtarg(i,name)
                read(name,'(bn,f10.0)')f3
                i = i + 1
                call mgtarg(i,name)
                read(name,'(bn,f10.0)')f4
                if(f4.gt.f3 .and. f3.gt.f2 .and. f2.gt.f1
     1              .and. f1.gt.0.0)then
                    doftaper = .true.
                else
                    doftaper = .false.
                    write(LER,*)'error in FREQLIMITS'
                endif
            else if(name(1:4).eq.'-AMP')then
                i = i + 1
                call mgtarg(i,famp)
            else if(name(1:4).eq.'-PHA')then
                i = i + 1
                call mgtarg(i,fpha)
            else if(name(1:5).eq.'-SACI')then
                i = i + 1
                call mgtarg(i,sacin)
            else if(name(1:5).eq.'-SACO')then
                i = i + 1
                call mgtarg(i,sacout)
            else if(name(1:2).eq.'-?')then
                call usage(' ')
            else if(name(1:2).eq.'-h')then
                call usage(' ')
            endif
        go to 11
   13   continue
        return
        end

        subroutine usage(str)
        parameter (LER=0, LIN=5, LOT=6)
        character str*(*)
        write(LER,*)'sacevalr:',str
        write(LER,*)'USAGE: ',
     1       'sacevalr -DEMEAN -TAPER -FREQLIMITS f1 f2 f3 f4 -A -R'
        write(LER,*)
     1  '       -AMP amp_file -PHA phase_file -SACIN binary_sac_in'
        write(LER,*)
     1  '       -SACOUT binary_sac_out'
        write(LER,*)
     1  ' -DEMEAN  (default false) remove mean before filter'
        write(LER,*)
     1  ' -TAPER   (default false) apply taper before filter'
        write(LER,*)
     1  ' -FREQLIMITS f1 f2 f3 f4 (default all pass)',
     2  '          Mute for f < f1, f > f4, taper between f1-f2',
     3  '          f3-f4, pass between f2-f3'
        write(LER,*)
     1  ' -A       (default true) apply filter'
        write(LER,*)
     1  ' -R       (default false) remove filter'
        write(LER,*)
     1  ' -AMP amp_file   (none)  evalresp amp file'
        write(LER,*)
     1  ' -PHA phase_file (none)  evalresp phase file'
        write(LER,*)
     1  ' -SACIN  binary_sac_input  file (none)'
        write(LER,*)
     1  ' -SACOUT binary_sac_output file (none)'
        write(LER,*)
     1  ' -?           Write this help message'
        write(LER,*)
     1  ' -h           Write this help message'
        write(LER,*)
     1  ' SAC header values set'
        write(LER,*)
     1  '  USER1 :  per_min       KUSER1:  PER_MIN'
        write(LER,*)
     1  '  USER2 :  per_max       KUSER2:  PER_MAX'
        stop
        end

        subroutine dmean(x,npts)
c-----
c      remove the mean from the time series.
c      Note this will usually remove the DC offset of the
c      trace, but for a one sided pulse this may seriously
c      affect the trace and its spectrum
c-----
c      x   R*4 - time series array
c      npts    I*4 - numbe of samples
c-----
        integer*4 npts
        real*4 x(npts)
        real*8 dsum
c-----
c      remove the mean
c-----
        if(npts.gt.1)then
            dsum  = 0.0d+00
            do 1000 i=1,npts
                dsum = dsum + dble(x(i))
 1000       continue
            dsum = dsum / npts
            xmean = sngl (dsum)
            do 2000 i= 1, npts
                x(i) = x(i) - xmean
 2000       continue
        endif
        return
        end

        subroutine dtape(x,npts,pct)
c-----
c      taper the edges of the trace
c      
c      x   R*4 - input time series
c      npts    I*4 - number of points
c      pct R*4 - percentage of taper
c                The taper covers pct percent of the total
c                length of the time series from each end
c          NOTE CHECK WITH SAC CONCEPT
c-----
        integer*4 npts
        real*4 x(npts), pct
        n1 = (pct * npts) / 100
        if(n1.le.1)return
c-----
c      perfrom the taper
c          x(1) = 0.0
c          x(n1) = 1.0
c          ...
c          x(npts - n1 + 1) = 1.0
c          x(npts) = 0.0
c-----
        x(1) = 0.0
        x(npts) = 0.0
        do 1000 i=2,n1
            call costap(i,n1,xx)
            x(i) = xx * x(i)
            x(npts-n1+i) = xx * x(npts-n1+i)
 1000   continue
        return
        end
        
        subroutine costap(i,n,xx)
c-----
c      provide a cosine taper
c-----
c      i   I*4 - pointer 1 <= i <= n
c      n   I*4 - 
c      xx  R*4 - taper function
c                  xx = 0.0 for i = 1
c                  xx = 1.0 for i = n
c                  xx = 1.0 - cos( (i-1)* pi / 2(n-1))
c-----
        pi2 = 3.1415927 / 2.0
c-----
c      safety check
c-----
        if(n.le.1)return
c-----
c      general taper function
c-----
        if(i .le. 1)then
            xx = 0.0
        else if(i.gt.1 .and. i.lt.n)then
            xx = 1.0 - cos( (i-1)*pi2/(n-1))
        else if(i.ge.n)then
            xx = 1.0
        endif
        return
        end

        subroutine filt(doappl,x,npts,dt,f1,f2,f3,f4,doftaper,
     1          nfreq,tfreq,tamp,tpha,NRESP,permin,permax)
c-----
c      doappl  L   - .true. apply filter
c                .false. remove filter with water level percentage
c      x   R*4 - time series array
c      npts    I*4 - number of points in time series
c      dt  R*4 - sampling interval
c      doftaper L  If true do frequency taper
c      f1  R*4 Apply zero phase band bass [f1,f2,f3,f4]
c      f2  R*4 where response = 1 for f2 <= f <= f3
c      f3  R*4
c      f4  R*4
c      nfreq   I   number of frequency values
c      tfreq   R*4 array of frequencies in increasing order
c      tamp    R*4 array of real amplitude response created
c      tpha    R*4 array of real phase     response created
c      NRESP   I   dimension of the arrays
c      permin  R*4 minimum period 
c      permax  R*4 maximum period of passband defined approximately as
c              frequencies where response is > 2 water*zmax or
c              2 zb
c-----
        integer*4 NSAMP
        parameter (NSAMP=131072)

        logical doappl
        integer*4 npts
        real*4 x(NSAMP)
        real dt
        real f1, f2, f3, f4
        logical doftaper
        real taper3
        real permin, permax
        

        integer NRESP
        real tamp(NRESP), tpha(NRESP)
        integer nfreq
        real tfreq(NRESP)

        common/zz/z(NSAMP)
        complex z

        real taper

        complex camp

c-----
c      get a power of 2
c-----
        n = npts
        call npow2(n)
        n21 = n /2 + 1
c-----
c      zero fill and get FFT
c-----
        do 1000 i=1,n
            if(i.le.npts)then
                z(i) = cmplx(x(i),0.0)
            else
                z(i) = cmplx(0.0,0.0)
            endif
 1000   continue
c-----
c      get Discrete Fourier transform
c-----
        call zfour(z,n,-1,dt,df)
c------
c      get the instrument response
c-----
        if(doappl)then
            do 2000 i=1,n21
                freq = (i-1)*df
            call resp(freq,camp,nfreq,tfreq,tamp,tpha,NRESP)
                z(i) = z(i) * camp
                if(i.gt.1)then
                    z(n+2 -i) = conjg(z(i))
                endif
 2000       continue
        else
c-----
c      deconvolve the filter response
c-----
            do 3000 i=1,n21
                if(i.eq.1)then
                    freq = 0.01*df
                else
                    freq = (i-1)*df
                endif
            call resp(freq,camp,nfreq,tfreq,tamp,tpha,NRESP)
                if(doftaper)then
                    if(freq.lt.f2)then
                        taper= taper3(f1,f2,freq)
                    else if(freq.gt.f3)then
                        taper= taper3(f4,f3,freq)
                    else 
                        taper = 1.0
                    endif
                else
                    taper = 1.0
                endif
                z(i) = z(i)*taper/camp
                if(i.gt.1)then
                    z(n+2 -i) = conjg(z(i))
                endif
 3000       continue
c-----
c          set frequency limits for the header
c-----
            if(doftaper)then
                    permax = 1.0/f2
                    permin = 1.0/f3
            else
                    permax = n*dt
                    permin = 1.0/(2.0*dt)
            endif
        endif
        z(n21) = cmplx(real(z(n21)),0.0)
        call zfour(z,n,+1,dt,df)
c-----
c      obtain the inverse FFT
c-----
        do 5000 i=1,npts
            x(i) = real(z(i))
 5000   continue
        return
        end

        subroutine npow2(npts)
c-----
c      Given npts, determine the N=2**m such that N >= npts
c      return the new ntps
c-----
        integer*4 nsamp, npts
        nsamp = npts
        npts = 1
 1000   continue
        npts = 2*npts
        if(npts.lt.nsamp)goto 1000
        return
        end

        subroutine zfour(zarr,nn,isign,dt,df) 
c-----
c     THE input is a complex array
c     which has numbers stored in memory as
c     R1, I1, R2, I2, ..., Rnn, Inn
c     where nn must be a power of 2 R and I are the real and imaginary
c     parts of the complex number
c
c     For isign -1 this is a complex time series
c     For isign +1 this is a complex frequency series with
c        index 1 (in fortran corresponding to f=0
c              2                              f=df
c            nn/2 + 1                         f = 1/2dt = Nyquist
c            nn - 1                           f = -2df
c             nn                              f = -df

c-----
c     the cooley-tookey fast fourier transform in usasi basic fortran
c     transform(j) = sum(datc(i)*w**((i-1)(j-1)), where i and j run
c     from 1 to nn and w = exp(isign*2*pi*sqrt(-1)/nn).  datc is a one-
c     dimensional complex array (i.e., the real and imaginary parts of
c     datc are located immediately adjacent in storage, such as fortran
c     places them) whose length nn is a power of two.  isign
c     is +1 or -1, giving the sign of the transform.  transform values
c     are returned in array datc, replacing the input datc.  the time is
c     proportional to n*log2(n), rather than the usual n**2
c     rms resolution error being bounded by 6*sqrt(i)*log2(nn)*2**(-b),
c     b is the number of bits in the floating point fraction.
c
c     the program computes df from dt, dt from df and checks to see
c     if they are consistent. In addition, the transforms are multiplied
c     by dt or df to make the results dimensionally correct
c
c     This is a slightly modified version of the original Brenner routine
c     The changes were to add physical dimensions to the transform
c     and to make it all complex
c-----
        complex zarr(*) 
        integer nn, isign
        real dt, df

        complex ztemp
c-----
c       ensure that the dt and df are defined and
c       consistent
c-----
        if(dt.eq.0.0) dt = 1./(nn*df) 
        if(df.eq.0.0) df = 1./(nn*dt) 
        if(dt.ne.(nn*df)) df = 1./(nn*dt) 
c-----
c       now begin the transform
c-----
        jj = 1
        do 5 ii=1,nn 
        if(ii .lt. jj) then
              ztemp = zarr(jj)
              zarr(jj) = zarr(ii)
              zarr(ii) = ztemp
        endif
        mm = nn/2
    3   continue
        if(jj.le.mm) then
            go to 55
        else 
              jj = jj - mm
              mm = mm /2
              if(mm.lt.1)then
                  go to 55
              else
                  go to 3
              endif
        endif
   55   continue
        jj = jj + mm
    5   continue
        mmax = 1 
c-----
    6 continue
        if(mmax .lt. nn)then
            go to 7
        else if(mmax .ge. nn)then
            go to 10
        endif
    7   continue
        istep= 2*mmax 
        theta = 6.283185307/(isign*2.0*mmax) 
        sinth=sin(theta/2.) 
        wstpr=-2.*sinth*sinth 
        wstpi=sin(theta) 
        wr=1.0 
        wi=0.0 
        do 9 mm=1,mmax
              do 8 ii=mm,nn,istep
                    jj=ii+mmax
                    ztemp=cmplx(wr,wi)*zarr(jj)
                    zarr(jj) = zarr(ii) - ztemp
                    zarr(ii) = zarr(ii) + ztemp
    8         continue
c-----
c       use trig relations to compute the next sin/cos
c       without actually calling sin() or cos()
c-----
              tempr = wr 
              wr = wr*wstpr-wi*wstpi + wr 
              wi = wi*wstpr+tempr*wstpi + wi 
    9   continue
        mmax = istep 
        go to 6 
c-----
c       transform is done
c-----
   10   continue 
c-----
c     give the arrays the proper physical dimensions
c-----
        if(isign.lt.0)then
c-----
c             time to frequency domain
c-----
              do  ii = 1,nn
                    zarr(ii) = zarr(ii) * dt
              enddo
        else
c-----
c             frequency to time domain
c-----
              do ii = 1,nn
                    zarr(ii) = zarr(ii) * df
              enddo
        endif
        return
        end

        subroutine gettamp(frespa,frespp,nfreq,tfreq,
     1      tamp,tpha,NRESP)
c-----
c      frespa  Ch*80   name of amplitude response file
c      frespp  Ch*80   name of phase     response file
c      nfreq   I   number of frequency values
c      tfreq   R*4 array of frequencies in increasing order
c      tamp    R*4 array of real amplitude response created
c      tpha    R*4 array of real phase     response created
c      NRESP   I   dimension of the arrays
c-----
        character frespa*(*)
        character frespp*(*)
        real tamp(NRESP), tpha(NRESP)
        real*4 tfreq(NRESP)
        logical ext
        parameter (LER=0, LIN=5, LOT=6)

c-----
c      see of the amplitude response file exists
c-----
        inquire(file=frespa,exist=ext)
        if (.not. ext) then
            call usage('Amplitude response file does not exist')
        endif
c-----
c      see of the phase response file exists
c-----
        inquire(file=frespp,exist=ext)
        if (.not. ext) then
            call usage('Phase response file does not exist')
        endif
c-----
c      open the response file
c      assume that both files are in the order of increasing frequency
c      and have eactly the same number of entries
c-----
        ls = lgstr(frespa)
        open(1,file=frespa(1:ls),form='formatted',
     1      status='unknown',access='sequential')
        rewind 1
        ls = lgstr(frespp)
        open(2,file=frespp(1:ls),form='formatted',
     1      status='unknown',access='sequential')
        rewind 2
        j = 0   
        pcorr = 0.0
 1000   continue
            read(1,*,end=9990,err=9990)fa,a
            read(2,*,end=9990,err=9990)fp,p
            j = j + 1
c-----
c      attempt to correct for 360 degree phase jump
c          this is not perfect
c-----
            if(j.eq.1)then
c-----
c              initialize
c-----
                oldpha = p
            else
c-----
c              test for phase jump
c-----
                if(abs(p-oldpha).gt.350.0)then
                    dp = p - oldpha
                    if(dp.gt.0.0)then
                        pcorr = pcorr - 360.0
                    else
                        pcorr = pcorr + 360.0
                    endif
                endif
            endif
            oldpha = p
            nfreq = j
            tfreq(j) = fa
            tamp(j) = a
            tpha(j) = p + pcorr
        go to 1000
 9990   continue
        close(1)
        close(2)
c-----
c      now examine the phase response and attempt to detect
c      the effects of 2 pi phase jumps
c-----
c      basically look examine slope, look for a change in slope
c      greater than almost 360 degrees or so, maintain the  than
c      add a sgn(slope)*360.0
c-----
        return
        end

        subroutine resp(freq,camp,nfreq,tfreq,tamp,tpha,NRESP)
c-----
c      get complex response for a given frequency by
c      interpolating the tables
c-----
c      freq    R*4 desired frequency
c      camp    C*8 complex response
c      nfreq   I   number of frequency values
c      tfreq   R*4 array of frequencies in increasing order
c      tamp    R*4 array of real amplitude response created
c      tpha    R*4 array of real phase     response created
c      NRESP   I   dimension of the arrays
c-----
        real*4 freq
        complex camp
        real tamp(NRESP), tpha(NRESP)
        real*4 tfreq(NRESP)

        real p 
c-----
c      assume order from least to largest in frequency
c-----
        do 1000 i=1,nfreq-1
            fup = amax1(tfreq(i),tfreq(i+1))
            flw = amin1(tfreq(i),tfreq(i+1))
            if(freq.ge.flw .and. freq.le.fup)then
                p = ( freq - tfreq(i))/(tfreq(i+1) - tfreq(i))
                amp = (1.0-p)*tamp(i) + p*tamp(i+1)
                pha = (1.0-p)*tpha(i) + p*tpha(i+1)
                call zresp(camp,  amp, pha)
            endif
 1000   continue
c-----
c      do end member test
c-----
        if(tfreq(1).lt.tfreq(nfreq))then
            if(freq.le.tfreq(1))then
                call zresp(camp,  tamp(1), tpha(1))
            endif
            if(freq.ge.tfreq(nfreq))then
                call zresp(camp,  tamp(nfreq), tpha(nfreq))
            endif
        else
            if(freq.ge.tfreq(1))then
                call zresp(camp,  tamp(1), tpha(1))
            endif
            if(freq.le.tfreq(nfreq))then
                call zresp(camp,  tamp(nfreq), tpha(nfreq))
            endif
        endif
        return
        end

        subroutine zresp(camp, amp, pha)
c-----
c      create a complex amplitude from an amplitude and phase(degrees)
c-----
c      camp    C*8 Complex response
c      amp R*4 amplitude response
c      pha R*4 phase response
c-----
        complex camp
        real amp, pha
        phase = (3.1415927/180.0)*pha
        camp = amp * cmplx(cos(phase),sin(phase))
        return
        end

        real function taper3(xl,xh,x)
        real xl, xh, x
        real p
c-----
c      cubic taper between 0 and 1 for xl <= x <= xh,
c      with = 1 for x = xh and 0 for x = xl,
c-----
        if(xl .eq. xh)then
            taper3 = 1.0
        else
            p = (x - xl)/(xh - xl)
            p = 2.0*p - 1.0
            if(p .le. -1.0)then
                taper3 = 0.0
            else if(p .ge. 1.0)then
                taper3 = 1.0
            else 
                taper3 = 0.5 + 0.75*p*(1.0 - p*p/3.0)
            endif
        endif
        return
        end
